package org.discordbot;

import net.dv8tion.jda.api.entities.Guild;
import net.dv8tion.jda.api.entities.GuildVoiceState;
import net.dv8tion.jda.api.entities.Member;
import net.dv8tion.jda.api.entities.User;
import net.dv8tion.jda.api.entities.channel.concrete.VoiceChannel;
import net.dv8tion.jda.api.events.interaction.command.SlashCommandInteractionEvent;
import net.dv8tion.jda.api.events.message.MessageReceivedEvent;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class App {
    public static ArrayList<Content> contentArrayList;
    private static final String GM_ID = "1256616828025835664";
    private static final String ADMIN_ID = "1263479088018882604";
    public static int countRecentContents(ArrayList<Content> contents) {
        LocalDateTime now = LocalDateTime.now();
        int count = 0;

        if (contents == null || contents.isEmpty()) {
            return 0;
        }

        for (Content content : contents) {
            LocalDateTime contentTime = content.getTime();
            if (contentTime != null && ChronoUnit.DAYS.between(contentTime, now) <= 7) {
                count++;
            }
        }

        return count;
    }

    public static void exportDataToSheets() throws GeneralSecurityException {

        Map<String, Player> players = JsonUtil.readPlayers();
        ArrayList<Content> contents = JsonUtil.readContents();

        List<List<Object>> playerData = DataConverter.convertPlayersToSheetFormat(players);
        List<List<Object>> contentData = DataConverter.convertContentsToSheetFormat(contents);

        SheetsService.writeDataToSheet(playerData, "Players");
        SheetsService.writeDataToSheet(contentData, "Contents");
    }

    public static String[] getVoiceChannelMemberIds(User user) {
        // Get the user's guilds and find their member object
        Member member = null;
        for (Guild guild : user.getJDA().getGuilds()) {
            Member guildMember = guild.getMember(user);
            if (guildMember != null) {
                member = guildMember;
                break;
            }
        }

        if (member == null) {
            return new String[0]; // Return an empty array if the member is not found
        }

        // Get the voice state of the member to check if they are in a voice channel
        GuildVoiceState voiceState = member.getVoiceState();
        if (voiceState == null || voiceState.getChannel() == null) {
            return new String[0]; // Return an empty array if the user is not in a voice channel
        }

        // Get the voice channel the member is in
        VoiceChannel voiceChannel = (VoiceChannel) voiceState.getChannel();

        // Get all members in the same voice channel
        List<Member> members = voiceChannel.getMembers();

        // Extract their IDs in mention format and convert them to a String array
        return members.stream()
                .map(m -> "<@" + m.getUser().getId() + ">")
                .toArray(String[]::new);
    }

    public static String messageReceived(MessageReceivedEvent event) {
        String message = event.getMessage().getContentRaw();

        if (message.startsWith("!attendance ")) {
            message = message.replace("!attendance ", "");

            if (message.isBlank())
                return "Please enter a name for the content";

            Map<String, Player> players = JsonUtil.readPlayers();
            ArrayList<Content> contents = JsonUtil.readContents();

            if (players == null) {
                players = new HashMap<>();
            }

            if (contents == null) {
                contents = new ArrayList<>();
            }

            contentArrayList = contents;

            Content tempContent = new Content(message, event.getMessage().getTimeCreated().format(DateTimeFormatter.ofPattern("HH:mm dd/MM/yyyy")), event.getAuthor().getId());

            String[] parts = getVoiceChannelMemberIds(event.getAuthor());

            if (parts == null || parts.length == 0){
                return  "No voice chat found";
            }

            for (int i = 0; i < parts.length; i++) {
                String playerId = parts[i].trim();
                tempContent.addPlayer(playerId);
            }

            for (int i = 0; i < parts.length; i++) {
                String playerId = parts[i].trim();

                Player player = players.get(playerId);
                if (player != null) {
                    player.addContent(tempContent);
                } else {
                    Player newPlayer = new Player(playerId);
                    newPlayer.addContent(tempContent);
                    players.put(playerId, newPlayer);
                }
            }

            tempContent.updatePlayerNames();
            contents.add(tempContent);
            JsonUtil.writeData(players, contents);

            try {
                exportDataToSheets();
            } catch (Exception e) {
                StackTraceElement element = e.getStackTrace()[0];
                int lineNumber = element.getLineNumber();
                String className = element.getClassName();
                String methodName = element.getMethodName();

                return  ("Something went wrong, please check your format, otherwise message a GM.\n"
                        + "Error Type: " + e.getClass()
                        + "\nOccurred in: " + className
                        + "." + methodName
                        + " at line: " + lineNumber);
            }

            return ("Successfully added content \"" + tempContent.getName() + "\".");
        } else if (message.startsWith("!playerinfo ")) {
            message = message.replace("!playerinfo ", "");

            if (message.isBlank())
                return "Please enter a name for information about the player";

            Map<String, Player> players = JsonUtil.readPlayers();
            ArrayList<Content> contents = JsonUtil.readContents();

            Player player = players.get(message.trim());
            if (player == null) {
                return "Player not found.";
            }
            String output = "Info about player " + player.getPlayerID() + ":\n" +
                    "Player participated in " + player.getContentsParticipated().size() + "/" + contents.size() + " contents.\n" +
                    "Player participated in " + countRecentContents(player.getContentsParticipated()) + "/" + countRecentContents(contents) + " contents in the last 7 days.";
            return (output);
        } else if (message.startsWith("!deleteplayer ")) {
            message = message.replace("!deleteplayer ", "");

            boolean isAdmin = event.getMember().getRoles().stream()
                    .anyMatch(role -> role.getId().equals(ADMIN_ID) || role.getId().equals(GM_ID));

            if (!isAdmin)
                return "You do not have permission to use this command.";

            if (message.isBlank())
                return "Please enter the name of the player you wish to delete";

            Map<String, Player> players = JsonUtil.readPlayers();
            ArrayList<Content> contents = JsonUtil.readContents();

            Player player = players.get(message.trim());
            if (player == null) {
                return "Player not found.";
            }

            players.remove(message.trim());
            JsonUtil.writeData(players, contents);
            return ("Succesfully deleted " + player.getPlayerID() + " from the database.");
        } else if (message.startsWith("!content ")) {
            message = message.replace("!content ", "");

            if (message.isBlank())
                return "Please check the format for !content";

            Map<String, Player> players = JsonUtil.readPlayers();
            ArrayList<Content> contents = JsonUtil.readContents();

            if (players == null) {
                players = new HashMap<>();
            }

            if (contents == null) {
                contents = new ArrayList<>();
            }

            contentArrayList = contents;

            String[] options = message.split(";");

            if (options.length != 4)
                return "Please check the format for !content";

            Content tempContent = new Content(options[0], options[1], options[2].trim());

            String[] parts = options[3].split(" ");

            String output = "";

            for (int i = 0; i < parts.length; i++) {
                String playerId = parts[i].trim();

                if (players.containsKey(playerId)) {
                    tempContent.addPlayer(playerId);
                } else if (!playerId.isBlank()) {
                    output += ("\nPlayer " + playerId + " did not get added to the content.");
                }
            }

            for (int i = 0; i < parts.length; i++) {
                String playerId = parts[i].trim();

                if (players.containsKey(playerId)) {
                    players.get(playerId).addContent(tempContent);
                }
            }

            tempContent.updatePlayerNames(players);
            contents.add(tempContent);
            JsonUtil.writeData(players, contents);
            try {
                exportDataToSheets();
            } catch (Exception e) {
                StackTraceElement element = e.getStackTrace()[0];
                int lineNumber = element.getLineNumber();
                String className = element.getClassName();
                String methodName = element.getMethodName();

                return  ("Something went wrong, please check your format, otherwise message a GM.\n"
                        + "Error Type: " + e.getClass()
                        + "\nOccurred in: " + className
                        + "." + methodName
                        + " at line: " + lineNumber);
            }
            return ("Successfully added content \"" + tempContent.getName() + "\"." + output);
        } else if (message.startsWith("!delcontent")) {
            // Only allow admins or GMs to delete content
            boolean isAdmin = event.getMember().getRoles().stream()
                    .anyMatch(role -> role.getId().equals(ADMIN_ID) || role.getId().equals(GM_ID));

            if (!(isAdmin || event.getMember().getId().equals("465126844585476096")))
                return "You do not have permission to delete content.";

            return deleteLastContent();
        } else if (message.equalsIgnoreCase("!update")) {
            update();
            return "Players' contents updated and spreadsheet refreshed.";
        }

        return null;
    }

    public static String deleteLastContent() {
        ArrayList<Content> contents = JsonUtil.readContents();
        Map<String, Player> players = JsonUtil.readPlayers();

        if (contents == null || contents.isEmpty()) {
            return "No content found to delete.";
        }

        // Get the last content added
        Content lastContent = contents.get(contents.size() - 1);
        contents.remove(contents.size() - 1); // Remove the last content from the list

        // Remove the last content from each player's history
        for (String playerId : lastContent.getParticipantIDS()) {
            Player player = players.get(playerId);
            if (player != null) {
                player.getContentsParticipated().remove(lastContent); // Remove content from the player's list

                // Remove the player if they now have 0 contents participated
                if (player.getContentsParticipated().isEmpty()) {
                    players.remove(playerId); // Remove the player from the player list
                }
            }
        }

        // Write the updated data back to the JSON files
        JsonUtil.writeData(players, contents);

        // Export the updated data to Google Sheets
        try {
            exportDataToSheets();
        } catch (Exception e) {
            StackTraceElement element = e.getStackTrace()[0];
            int lineNumber = element.getLineNumber();
            String className = element.getClassName();
            String methodName = element.getMethodName();

            return "Something went wrong while deleting content.\n"
                    + "Error Type: " + e.getClass()
                    + "\nOccurred in: " + className
                    + "." + methodName
                    + " at line: " + lineNumber;
        }

        return "Successfully deleted the last content: \"" + lastContent.getName() + "\" and cleaned up players with no content.";
    }

    public static void update() {
        // Step 1: Get the global list of contents and the list of players
        ArrayList<Content> globalContents = JsonUtil.readContents(); // Read the global contents
        Map<String, Player> players = JsonUtil.readPlayers(); // Read the player data

        // Step 2: For each player, check their contents and remove any that are not in the global list
        for (Player player : players.values()) {
            ArrayList<Content> playerContents = player.getContentsParticipated();

            // Use removeIf to remove any content not found in the global contents list
            playerContents.removeIf(content -> !globalContents.contains(content));
        }

        // Step 3: Write the updated player data back to the JSON file
        JsonUtil.writePlayers(players);

        // Step 4: Update the Google Sheets spreadsheet to reflect the changes
        SheetsService.writeDataToSheet(DataConverter.convertPlayersToSheetFormat(players), "Players");
        SheetsService.writeDataToSheet(DataConverter.convertContentsToSheetFormat(globalContents), "Contents");

        System.out.println("Update complete: invalid contents removed from players, and the sheet has been updated.");
    }
}
