import threading
import random
import string
from selenium import webdriver
from selenium.webdriver.chrome.options import Options
from PIL import Image, ImageDraw
import os
import time
import shutil
import csv
from collections import deque
import math
import datetime

# Clear folders
for folder in ["bathingpools/screenshots", "bathingpools/movement", "bathingpools/movement_image", "bathingpools/clusters"]:
    if os.path.exists(folder):
        for filename in os.listdir(folder):
            file_path = os.path.join(folder, filename)
            if os.path.isfile(file_path) or os.path.islink(file_path):
                os.unlink(file_path)
            elif os.path.isdir(file_path):
                shutil.rmtree(file_path)

SCREENSHOT_ID = 1

def get_id():
    global SCREENSHOT_ID
    return SCREENSHOT_ID

def set_id(id):
    global SCREENSHOT_ID
    SCREENSHOT_ID = id

class RandomNumberGenerator(threading.Thread):
    def __init__(self):
        super().__init__()
        self.daemon = True
        self.take_screenshot_event = threading.Event()
        self.screenshot_done_event = threading.Event()
        self.screenshot_path = None

        chrome_options = Options()
        chrome_options.add_argument("--headless=new")
        chrome_options.add_argument("--disable-gpu")
        chrome_options.add_argument("--window-size=1280,800")
        chrome_options.add_argument("--hide-scrollbars")
        chrome_options.add_argument("--force-device-scale-factor=1")

        self.driver = webdriver.Chrome(options=chrome_options)
        self.driver.get("https://thebathingpools.com")
        time.sleep(2)
        self.driver.execute_script("window.scrollBy(0, 900);")
        time.sleep(4)

    def run(self):
        while True:
            self.take_screenshot_event.wait()
            self.take_screenshot_event.clear()

            path = self.screenshot_path
            os.makedirs(os.path.dirname(path), exist_ok=True)
            self.driver.save_screenshot(path)

            image = Image.open(path)
            width, height = image.size
            cropped = image.crop((300, 300, width - 340, height - 230))
            cropped.save(path)

            self.screenshot_done_event.set()

    def cluster(self, show_logs=False):
        current_id = get_id()
        next_id = current_id + 1

        path1 = f"bathingpools/screenshots/{current_id}.png"
        self.screenshot_path = path1
        self.take_screenshot_event.set()
        self.screenshot_done_event.wait()
        self.screenshot_done_event.clear()
        
        time.sleep(1)

        path2 = f"bathingpools/screenshots/{next_id}.png"
        self.screenshot_path = path2
        self.take_screenshot_event.set()
        self.screenshot_done_event.wait()
        self.screenshot_done_event.clear()

        if not os.path.exists(path1) or not os.path.exists(path2):
            print("[!] Failed to take one or both screenshots.")
            return

        digit = self.detect_motion_image(path1, path2, current_id, next_id, show_logs=show_logs)
        set_id(current_id + 2)
        return digit

    def detect_motion_image(self, image_path1, image_path2, id1, id2, dest_folder="bathingpools/movement", show_logs=False):
        image1 = Image.open(image_path1).convert("RGB")
        image2 = Image.open(image_path2).convert("RGB")
        width, height = image1.size

        delta_threshold = 250
        min_delta_threshold = 10

        clusters = []
        filtered_matrix = []

        while delta_threshold >= min_delta_threshold:
            motion_matrix = []
            for y in range(height):
                row = []
                for x in range(width):
                    r1, g1, b1 = image1.getpixel((x, y))
                    r2, g2, b2 = image2.getpixel((x, y))
                    delta = abs(r1 - r2) + abs(g1 - g2) + abs(b1 - b2)
                    row.append(1 if delta > delta_threshold else 0)
                motion_matrix.append(row)

            def neighbor_coords(y, x):
                for dy in (-1, 0, 1):
                    for dx in (-1, 0, 1):
                        if dy == 0 and dx == 0:
                            continue
                        ny, nx = y + dy, x + dx
                        if 0 <= ny < height and 0 <= nx < width:
                            yield ny, nx

            def is_core_pixel(y, x):
                return sum(
                    motion_matrix[ny][nx]
                    for ny, nx in neighbor_coords(y, x)
                ) >= 4

            filtered_matrix = []
            for y in range(height):
                row = []
                for x in range(width):
                    if motion_matrix[y][x] == 1:
                        if is_core_pixel(y, x):
                            row.append(1)
                        else:
                            if any(is_core_pixel(ny, nx) and motion_matrix[ny][nx] == 1 for ny, nx in neighbor_coords(y, x)):
                                row.append(1)
                            else:
                                row.append(0)
                    else:
                        row.append(0)
                filtered_matrix.append(row)

            clusters = self.get_clusters(filtered_matrix)
            large_clusters = [c for c in clusters if len(c) > 15]

            if len(large_clusters) > 5 and len(large_clusters) < 20:
                break
            delta_threshold -= 10

        os.makedirs(dest_folder, exist_ok=True)
        motion_csv_path = os.path.join(dest_folder, f"motion_{id1}_to_{id2}.csv")
        with open(motion_csv_path, "w", newline="") as f:
            writer = csv.writer(f)
            writer.writerows(filtered_matrix)
        if show_logs:
            print(f"[✓] Motion CSV saved: {motion_csv_path}")

        os.makedirs("bathingpools/movement_image", exist_ok=True)
        motion_img = Image.new("RGB", (width, height), (255, 255, 255))
        motion_pixels = motion_img.load()
        for y in range(height):
            for x in range(width):
                if filtered_matrix[y][x] == 1:
                    motion_pixels[x, y] = (255, 0, 0)
        motion_img_path = f"bathingpools/movement_image/motion_image_{id1}_to_{id2}.png"
        motion_img.save(motion_img_path)
        if show_logs:
            print(f"[✓] Motion image saved: {motion_img_path}")

        # Draw clusters and get centers
        centers = self.print_clusters(motion_csv_path, motion_img_path, id1, id2, min_cluster_size=15, show_logs=show_logs)

        # Draw black lines between all cluster centers and calculate total distance
        motion_img = Image.open(motion_img_path)
        draw = ImageDraw.Draw(motion_img)
        total_dist = 0.0
        for i in range(len(centers)):
            for j in range(i + 1, len(centers)):
                x1, y1 = centers[i]
                x2, y2 = centers[j]
                total_dist += math.sqrt((x2 - x1) ** 2 + (y2 - y1) ** 2)
                draw.line([(x1, y1), (x2, y2)], fill="black", width=1)

        motion_img.save(motion_img_path)
        if show_logs:
            print(f"[✓] Clusters marked on image: {motion_img_path}")

        digit = int(total_dist) % 10
        if int(total_dist) == 0:
            digit = -1
        print(f"Total distance: {total_dist}")
        if show_logs:
            print(f"[✓] Generated random digit: {digit}")
        
        return digit

    def get_clusters(self, matrix):
        height = len(matrix)
        width = len(matrix[0]) if height > 0 else 0
        visited = [[False]*width for _ in range(height)]
        directions = [(-1,-1), (-1,0), (-1,1), (0,-1), (0,1), (1,-1), (1,0), (1,1)]
        clusters = []
        for y in range(height):
            for x in range(width):
                if matrix[y][x] == 1 and not visited[y][x]:
                    queue = deque([(y, x)])
                    visited[y][x] = True
                    cluster_points = []
                    while queue:
                        cy, cx = queue.popleft()
                        cluster_points.append((cx, cy))
                        for dy, dx in directions:
                            ny, nx = cy + dy, cx + dx
                            if 0 <= ny < height and 0 <= nx < width and matrix[ny][nx] == 1 and not visited[ny][nx]:
                                visited[ny][nx] = True
                                queue.append((ny, nx))
                    clusters.append(cluster_points)
        return clusters

    def print_clusters(self, csv_path, img_path, id1, id2, min_cluster_size=0, show_logs=False):
        with open(csv_path, newline="") as f:
            reader = csv.reader(f)
            matrix = [list(map(int, row)) for row in reader]

        clusters = self.get_clusters(matrix)
        motion_img = Image.open(img_path)
        draw = ImageDraw.Draw(motion_img)
        centers = []

        log_lines = []
        cluster_id = 1
        for cluster in clusters:
            if len(cluster) <= min_cluster_size:
                continue
            xs = [p[0] for p in cluster]
            ys = [p[1] for p in cluster]
            center_x = sum(xs) / len(xs)
            center_y = sum(ys) / len(ys)
            centers.append((center_x, center_y))

            log_lines.append(f"Cluster {cluster_id}: Center at (x={center_x:.2f}, y={center_y:.2f}), size={len(cluster)}")
            cx, cy = int(center_x), int(center_y)
            draw.line((cx - 5, cy, cx + 5, cy), fill="green", width=2)
            draw.line((cx, cy - 5, cx, cy + 5), fill="green", width=2)
            hull = self.convex_hull(cluster)
            if len(hull) > 2:
                draw.polygon(hull, outline="blue")
            cluster_id += 1

        os.makedirs("bathingpools/clusters", exist_ok=True)
        cluster_txt_path = f"bathingpools/clusters/clusters_{id1}_to_{id2}.txt"
        with open(cluster_txt_path, "w") as f:
            f.write("\n".join(log_lines))
        if show_logs:
            print(f"[✓] Cluster centers saved: {cluster_txt_path}")

        motion_img.save(img_path)
        if show_logs:
            print(f"[✓] Clusters marked on image: {img_path}")

        return centers

    def convex_hull(self, points):
        points = sorted(set(points))
        def cross(o, a, b):
            return (a[0]-o[0])*(b[1]-o[1]) - (a[1]-o[1])*(b[0]-o[0])
        lower = []
        for p in points:
            while len(lower) >= 2 and cross(lower[-2], lower[-1], p) <= 0:
                lower.pop()
            lower.append(p)
        upper = []
        for p in reversed(points):
            while len(upper) >= 2 and cross(upper[-2], upper[-1], p) <= 0:
                upper.pop()
            upper.append(p)
        return lower[:-1] + upper[:-1]

# usage

def append_number(number, csv_path="bathingpools/numbers.csv"):
    now = datetime.datetime.now()
    hour = now.hour
    minute = now.minute
    second = now.second

    os.makedirs(os.path.dirname(csv_path), exist_ok=True)
    file_exists = os.path.isfile(csv_path)

    with open(csv_path, "a", newline="") as f:
        writer = csv.writer(f)
        if not file_exists:
            writer.writerow(["hour", "minute", "second", "generated number"])
        writer.writerow([hour, minute, second, number])
        print(f"Time: {hour}:{minute}:{second} | Number: {number}")

random_generator = RandomNumberGenerator()
random_generator.start()
print("Wait for capture to load everything properly...")
time.sleep(10)
print("\\-> Done")

while True:
    print(f"IDs: {get_id()}, {get_id()+1}")
    digit = random_generator.cluster()
    append_number(digit)
    time.sleep(3)
