#!/usr/bin/env bash
set -euo pipefail

# ===== Config =====

BASE_DIR= "path_to_your_project_directory"  # TODO
DATA_DIR="$BASE_DIR/data"
OUTPUT_DIR="$BASE_DIR/output"
OUTPUT_FILE="$OUTPUT_DIR/combined.csv"

mkdir -p "$OUTPUT_DIR"

# ===== Functions =====

get_latest_data_dir() {
  ls -td "$DATA_DIR"/*/ | head -n 1
}

extract_timestamp_parts() {
  local timestamp_dir=$1
  local date_raw=${timestamp_dir%-*}
  local time_raw=${timestamp_dir#*-}

  DATE="${date_raw:0:4}-${date_raw:4:2}-${date_raw:6:2}"
  TIME="${time_raw:0:2}:${time_raw:2:2}"
}

transfrom_weather_data() {
  local weather_json=$1

  Temp_c=$(jq -r '.current.temp_c' "$weather_json")
  Wind_kph=$(jq -r '.current.wind_kph' "$weather_json")
  Humidity=$(jq -r '.current.humidity' "$weather_json")
  Cloud=$(jq -r '.current.cloud' "$weather_json")
}

transfrom_sail_data() {
  local sailings_json=$1

  Amount_boats=$(jq '[.[] | select(.Type == "Arrival")] | length' "$sailings_json")
  Popular_boat=$(jq -r '[.[] | select(.Type == "Arrival") | .Source] | group_by(.) | max_by(length) | .[0]' "$sailings_json" || echo "N/A")
}

transfrom_flight_data() {
  local flights_json=$1

  Amount_planes=$(jq 'length' "$flights_json")
  Popular_plane=$(jq -r '[.[].Source] | group_by(.) | max_by(length) | .[0]' "$flights_json" || echo "N/A")
}

append_to_csv() {
  if [ ! -f "$OUTPUT_FILE" ]; then
    echo "Date,Time,Temp_c,Wind_kph,Humidity,Cloud,Amount_boats,Amount_planes,Popular_boat,Popular_plane" > "$OUTPUT_FILE"
  fi

  echo "$DATE,$TIME,$Temp_c,$Wind_kph,$Humidity,$Cloud,$Amount_boats,$Amount_planes,$Popular_boat,$Popular_plane" >> "$OUTPUT_FILE"
}

# ===== Main =====

main() {
  local latest_dir
  latest_dir=$(get_latest_data_dir)

  TIMESTAMP_DIR="$(basename "$latest_dir")"
  WEATHER_JSON="$latest_dir/weather.json"
  SAILINGS_JSON="$latest_dir/sailings.json"
  FLIGHTS_JSON="$latest_dir/flights.json"

  extract_timestamp_parts "$TIMESTAMP_DIR"
  transfrom_weather_data "$WEATHER_JSON"
  transfrom_sail_data "$SAILINGS_JSON"
  transfrom_flight_data "$FLIGHTS_JSON"
  append_to_csv
}

main "$@"
